<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\VarDumper\Tests\Caster;

use PHPUnit\Framework\TestCase;
use Symfony\Component\VarDumper\Caster\ArgsStub;
use Symfony\Component\VarDumper\Caster\ClassStub;
use Symfony\Component\VarDumper\Caster\LinkStub;
use Symfony\Component\VarDumper\Cloner\VarCloner;
use Symfony\Component\VarDumper\Dumper\HtmlDumper;
use Symfony\Component\VarDumper\Test\VarDumperTestTrait;
use Symfony\Component\VarDumper\Tests\Fixtures\FooInterface;

class StubCasterTest extends TestCase
{
    use VarDumperTestTrait;

    public function testArgsStubWithDefaults($foo = 234, $bar = 456)
    {
        $args = [new ArgsStub([123], __FUNCTION__, __CLASS__)];

        $expectedDump = <<<'EODUMP'
array:1 [
  0 => {
    $foo: 123
  }
]
EODUMP;

        $this->assertDumpMatchesFormat($expectedDump, $args);
    }

    public function testArgsStubWithExtraArgs($foo = 234)
    {
        $args = [new ArgsStub([123, 456], __FUNCTION__, __CLASS__)];

        $expectedDump = <<<'EODUMP'
array:1 [
  0 => {
    $foo: 123
    ...: {
      456
    }
  }
]
EODUMP;

        $this->assertDumpMatchesFormat($expectedDump, $args);
    }

    public function testArgsStubNoParamWithExtraArgs()
    {
        $args = [new ArgsStub([123], __FUNCTION__, __CLASS__)];

        $expectedDump = <<<'EODUMP'
array:1 [
  0 => {
    123
  }
]
EODUMP;

        $this->assertDumpMatchesFormat($expectedDump, $args);
    }

    public function testArgsStubWithClosure()
    {
        $args = [new ArgsStub([123], '{closure}', null)];

        $expectedDump = <<<'EODUMP'
array:1 [
  0 => {
    123
  }
]
EODUMP;

        $this->assertDumpMatchesFormat($expectedDump, $args);
    }

    public function testLinkStub()
    {
        $var = [new LinkStub(__CLASS__, 0, __FILE__)];

        $cloner = new VarCloner();
        $dumper = new HtmlDumper();
        $dumper->setDumpHeader('<foo></foo>');
        $dumper->setDumpBoundaries('<bar>', '</bar>');
        $dumper->setDisplayOptions(['fileLinkFormat' => '%f:%l']);
        $dump = $dumper->dump($cloner->cloneVar($var), true);

        $expectedDump = <<<'EODUMP'
<foo></foo><bar><span class=sf-dump-note>array:1</span> [<samp>
  <span class=sf-dump-index>0</span> => "<a href="%sStubCasterTest.php:0" rel="noopener noreferrer"><span class=sf-dump-str title="55 characters">Symfony\Component\VarDumper\Tests\Caster\StubCasterTest</span></a>"
</samp>]
</bar>
EODUMP;

        $this->assertStringMatchesFormat($expectedDump, $dump);
    }

    public function testLinkStubWithNoFileLink()
    {
        $var = [new LinkStub('example.com', 0, 'http://example.com')];

        $cloner = new VarCloner();
        $dumper = new HtmlDumper();
        $dumper->setDumpHeader('<foo></foo>');
        $dumper->setDumpBoundaries('<bar>', '</bar>');
        $dumper->setDisplayOptions(['fileLinkFormat' => '%f:%l']);
        $dump = $dumper->dump($cloner->cloneVar($var), true);

        $expectedDump = <<<'EODUMP'
<foo></foo><bar><span class=sf-dump-note>array:1</span> [<samp>
  <span class=sf-dump-index>0</span> => "<a href="http://example.com" target="_blank" rel="noopener noreferrer"><span class=sf-dump-str title="11 characters">example.com</span></a>"
</samp>]
</bar>
EODUMP;

        $this->assertStringMatchesFormat($expectedDump, $dump);
    }

    public function testClassStub()
    {
        $var = [new ClassStub('hello', [FooInterface::class, 'foo'])];

        $cloner = new VarCloner();
        $dumper = new HtmlDumper();
        $dumper->setDumpHeader('<foo></foo>');
        $dumper->setDumpBoundaries('<bar>', '</bar>');
        $dump = $dumper->dump($cloner->cloneVar($var), true, ['fileLinkFormat' => '%f:%l']);

        $expectedDump = <<<'EODUMP'
<foo></foo><bar><span class=sf-dump-note>array:1</span> [<samp>
  <span class=sf-dump-index>0</span> => "<a href="%sFooInterface.php:10" rel="noopener noreferrer"><span class=sf-dump-str title="39 characters">hello(?stdClass $a, stdClass $b = null)</span></a>"
</samp>]
</bar>
EODUMP;

        $this->assertStringMatchesFormat($expectedDump, $dump);
    }

    public function testClassStubWithNotExistingClass()
    {
        $var = [new ClassStub(NotExisting::class)];

        $cloner = new VarCloner();
        $dumper = new HtmlDumper();
        $dumper->setDumpHeader('<foo></foo>');
        $dumper->setDumpBoundaries('<bar>', '</bar>');
        $dump = $dumper->dump($cloner->cloneVar($var), true);

        $expectedDump = <<<'EODUMP'
<foo></foo><bar><span class=sf-dump-note>array:1</span> [<samp>
  <span class=sf-dump-index>0</span> => "<span class=sf-dump-str title="Symfony\Component\VarDumper\Tests\Caster\NotExisting
52 characters"><span class="sf-dump-ellipsis sf-dump-ellipsis-class">Symfony\Component\VarDumper\Tests\Caster</span><span class=sf-dump-ellipsis>\</span>NotExisting</span>"
</samp>]
</bar>
EODUMP;

        $this->assertStringMatchesFormat($expectedDump, $dump);
    }

    public function testClassStubWithNotExistingMethod()
    {
        $var = [new ClassStub('hello', [FooInterface::class, 'missing'])];

        $cloner = new VarCloner();
        $dumper = new HtmlDumper();
        $dumper->setDumpHeader('<foo></foo>');
        $dumper->setDumpBoundaries('<bar>', '</bar>');
        $dump = $dumper->dump($cloner->cloneVar($var), true, ['fileLinkFormat' => '%f:%l']);

        $expectedDump = <<<'EODUMP'
<foo></foo><bar><span class=sf-dump-note>array:1</span> [<samp>
  <span class=sf-dump-index>0</span> => "<a href="%sFooInterface.php:5" rel="noopener noreferrer"><span class=sf-dump-str title="5 characters">hello</span></a>"
</samp>]
</bar>
EODUMP;

        $this->assertStringMatchesFormat($expectedDump, $dump);
    }

    public function testClassStubWithAnonymousClass()
    {
        $var = [new ClassStub(\get_class(new class() extends \Exception {
        }))];

        $cloner = new VarCloner();
        $dumper = new HtmlDumper();
        $dumper->setDumpHeader('<foo></foo>');
        $dumper->setDumpBoundaries('<bar>', '</bar>');
        $dump = $dumper->dump($cloner->cloneVar($var), true, ['fileLinkFormat' => '%f:%l']);

        $expectedDump = <<<'EODUMP'
<foo></foo><bar><span class=sf-dump-note>array:1</span> [<samp>
  <span class=sf-dump-index>0</span> => "<a href="%sStubCasterTest.php:195" rel="noopener noreferrer"><span class=sf-dump-str title="19 characters">Exception@anonymous</span></a>"
</samp>]
</bar>
EODUMP;

        $this->assertStringMatchesFormat($expectedDump, $dump);
    }
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             <?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\VarDumper\Tests\Caster;

use PHPUnit\Framework\TestCase;
use Symfony\Component\VarDumper\Test\VarDumperTestTrait;

/**
 * @author Baptiste Clavié <clavie.b@gmail.com>
 */
class XmlReaderCasterTest extends TestCase
{
    use VarDumperTestTrait;

    /** @var \XmlReader */
    private $reader;

    protected function setUp()
    {
        $this->reader = new \XmlReader();
        $this->reader->open(__DIR__.'/../Fixtures/xml_reader.xml');
    }

    protected function tearDown()
    {
        $this->reader->close();
    }

    public function testParserProperty()
    {
        $this->reader->setParserProperty(\XMLReader::SUBST_ENTITIES, true);

        $expectedDump = <<<'EODUMP'
XMLReader {
  +nodeType: NONE
  parserProperties: {
    SUBST_ENTITIES: true
     …3
  }
   …12
}
EODUMP;

        $this->assertDumpMatchesFormat($expectedDump, $this->reader);
    }

    /**
     * @dataProvider provideNodes
     */
    public function testNodes($seek, $expectedDump)
    {
        while ($seek--) {
            $this->reader->read();
        }
        $this->assertDumpMatchesFormat($expectedDump, $this->reader);
    }

    public function provideNodes()
    {
        return [
            [0, <<<'EODUMP'
XMLReader {
  +nodeType: NONE
   …13
}
EODUMP
            ],
            [1, <<<'EODUMP'
XMLReader {
  +localName: "foo"
  +nodeType: ELEMENT
  +baseURI: "%sxml_reader.xml"
   …11
}
EODUMP
            ],
            [2, <<<'EODUMP'
XMLReader {
  +localName: "#text"
  +nodeType: SIGNIFICANT_WHITESPACE
  +depth: 1
  +value: """
    \n
        
    """
  +baseURI: "%sxml_reader.xml"
   …9
}
EODUMP
            ],
            [3, <<<'EODUMP'
XMLReader {
  +localName: "bar"
  +nodeType: ELEMENT
  +depth: 1
  +baseURI: "%sxml_reader.xml"
   …10
}
EODUMP
            ],
            [4, <<<'EODUMP'
XMLReader {
  +localName: "bar"
  +nodeType: END_ELEMENT
  +depth: 1
  +baseURI: "%sxml_reader.xml"
   …10
}
EODUMP
            ],
            [6, <<<'EODUMP'
XMLReader {
  +localName: "bar"
  +nodeType: ELEMENT
  +depth: 1
  +isEmptyElement: true
  +baseURI: "%sxml_reader.xml"
   …9
}
EODUMP
            ],
            [9, <<<'EODUMP'
XMLReader {
  +localName: "#text"
  +nodeType: TEXT
  +depth: 2
  +value: "With text"
  +baseURI: "%sxml_reader.xml"
   …9
}
EODUMP
            ],
            [12, <<<'EODUMP'
XMLReader {
  +localName: "bar"
  +nodeType: ELEMENT
  +depth: 1
  +attributeCount: 2
  +baseURI: "%sxml_reader.xml"
   …9
}
EODUMP
            ],
            [13, <<<'EODUMP'
XMLReader {
  +localName: "bar"
  +nodeType: END_ELEMENT
  +depth: 1
  +baseURI: "%sxml_reader.xml"
   …10
}
EODUMP
            ],
            [15, <<<'EODUMP'
XMLReader {
  +localName: "bar"
  +nodeType: ELEMENT
  +depth: 1
  +attributeCount: 1
  +baseURI: "%sxml_reader.xml"
   …9
}
EODUMP
            ],
            [16, <<<'EODUMP'
XMLReader {
  +localName: "#text"
  +nodeType: SIGNIFICANT_WHITESPACE
  +depth: 2
  +value: """
    \n
            
    """
  +baseURI: "%sxml_reader.xml"
   …9
}
EODUMP
            ],
            [17, <<<'EODUMP'
XMLReader {
  +localName: "baz"
  +prefix: "baz"
  +nodeType: ELEMENT
  +depth: 2
  +namespaceURI: "http://symfony.com"
  +baseURI: "%sxml_reader.xml"
   …8
}
EODUMP
            ],
            [18, <<<'EODUMP'
XMLReader {
  +localName: "baz"
  +prefix: "baz"
  +nodeType: END_ELEMENT
  +depth: 2
  +namespaceURI: "http://symfony.com"
  +baseURI: "%sxml_reader.xml"
   …8
}
EODUMP
            ],
            [19, <<<'EODUMP'
XMLReader {
  +localName: "#text"
  +nodeType: SIGNIFICANT_WHITESPACE
  +depth: 2
  +value: """
    \n
        
    """
  +baseURI: "%sxml_reader.xml"
   …9
}
EODUMP
            ],
            [21, <<<'EODUMP'
XMLReader {
  +localName: "#text"
  +nodeType: SIGNIFICANT_WHITESPACE
  +depth: 1
  +value: "\n"
  +baseURI: "%sxml_reader.xml"
   …9
}
EODUMP
            ],
            [22, <<<'EODUMP'
XMLReader {
  +localName: "foo"
  +nodeType: END_ELEMENT
  +baseURI: "%sxml_reader.xml"
   …11
}
EODUMP
            ],
        ];
    }
}
