<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\VarDumper\Tests\Caster;

use PHPUnit\Framework\TestCase;
use Symfony\Component\VarDumper\Test\VarDumperTestTrait;

/**
 * @author Nicolas Grekas <p@tchwork.com>
 * @requires extension redis
 */
class RedisCasterTest extends TestCase
{
    use VarDumperTestTrait;

    public function testNotConnected()
    {
        $redis = new \Redis();

        $xCast = <<<'EODUMP'
Redis {
  isConnected: false
}
EODUMP;

        $this->assertDumpMatchesFormat($xCast, $redis);
    }

    public function testConnected()
    {
        $redis = new \Redis();
        if (!@$redis->connect('127.0.0.1')) {
            $e = error_get_last();
            self::markTestSkipped($e['message']);
        }

        $xCast = <<<'EODUMP'
Redis {%A
  isConnected: true
  host: "127.0.0.1"
  port: 6379
  auth: null
  mode: ATOMIC
  dbNum: 0
  timeout: 0.0
  lastError: null
  persistentId: null
  options: {
    TCP_KEEPALIVE: 0
    READ_TIMEOUT: 0.0
    COMPRESSION: NONE
    SERIALIZER: NONE
    PREFIX: null
    SCAN: NORETRY
  }
}
EODUMP;

        $this->assertDumpMatchesFormat($xCast, $redis);
    }
}
                                                                                                                                                                                        <?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\VarDumper\Tests\Caster;

use PHPUnit\Framework\TestCase;
use Symfony\Component\VarDumper\Test\VarDumperTestTrait;

/**
 * @author Jan Schädlich <jan.schaedlich@sensiolabs.de>
 */
class MemcachedCasterTest extends TestCase
{
    use VarDumperTestTrait;

    public function testCastMemcachedWithDefaultOptions()
    {
        if (!class_exists('Memcached')) {
            $this->markTestSkipped('Memcached not available');
        }

        $var = new \Memcached();
        $var->addServer('127.0.0.1', 11211);
        $var->addServer('127.0.0.2', 11212);

        $expected = <<<EOTXT
Memcached {
  servers: array:2 [
    0 => array:3 [
      "host" => "127.0.0.1"
      "port" => 11211
      "type" => "TCP"
    ]
    1 => array:3 [
      "host" => "127.0.0.2"
      "port" => 11212
      "type" => "TCP"
    ]
  ]
  options: {}
}
EOTXT;
        $this->assertDumpEquals($expected, $var);
    }

    public function testCastMemcachedWithCustomOptions()
    {
        if (!class_exists('Memcached')) {
            $this->markTestSkipped('Memcached not available');
        }

        $var = new \Memcached();
        $var->addServer('127.0.0.1', 11211);
        $var->addServer('127.0.0.2', 11212);

        // set a subset of non default options to test boolean, string and integer output
        $var->setOption(\Memcached::OPT_COMPRESSION, false);
        $var->setOption(\Memcached::OPT_PREFIX_KEY, 'pre');
        $var->setOption(\Memcached::OPT_DISTRIBUTION, \Memcached::DISTRIBUTION_CONSISTENT);

        $expected = <<<'EOTXT'
Memcached {
  servers: array:2 [
    0 => array:3 [
      "host" => "127.0.0.1"
      "port" => 11211
      "type" => "TCP"
    ]
    1 => array:3 [
      "host" => "127.0.0.2"
      "port" => 11212
      "type" => "TCP"
    ]
  ]
  options: {
    OPT_COMPRESSION: false
    OPT_PREFIX_KEY: "pre"
    OPT_DISTRIBUTION: 1
  }
}
EOTXT;

        $this->assertDumpEquals($expected, $var);
    }
}
