<?php

class One
{
    public $arg1;
    public $arg2;

    public function __construct($arg1 = null, $arg2 = null)
    {
        $this->arg1 = $arg1;
        $this->arg2 = $arg2;
    }
}

class Swift_DependencyContainerTest extends \PHPUnit\Framework\TestCase
{
    private $container;

    protected function setUp()
    {
        $this->container = new Swift_DependencyContainer();
    }

    public function testRegisterAndLookupValue()
    {
        $this->container->register('foo')->asValue('bar');
        $this->assertEquals('bar', $this->container->lookup('foo'));
    }

    public function testHasReturnsTrueForRegisteredValue()
    {
        $this->container->register('foo')->asValue('bar');
        $this->assertTrue($this->container->has('foo'));
    }

    public function testHasReturnsFalseForUnregisteredValue()
    {
        $this->assertFalse($this->container->has('foo'));
    }

    public function testRegisterAndLookupNewInstance()
    {
        $this->container->register('one')->asNewInstanceOf('One');
        $this->assertInstanceOf('One', $this->container->lookup('one'));
    }

    public function testHasReturnsTrueForRegisteredInstance()
    {
        $this->container->register('one')->asNewInstanceOf('One');
        $this->assertTrue($this->container->has('one'));
    }

    public function testNewInstanceIsAlwaysNew()
    {
        $this->container->register('one')->asNewInstanceOf('One');
        $a = $this->container->lookup('one');
        $b = $this->container->lookup('one');
        $this->assertEquals($a, $b);
    }

    public function testRegisterAndLookupSharedInstance()
    {
        $this->container->register('one')->asSharedInstanceOf('One');
        $this->assertInstanceOf('One', $this->container->lookup('one'));
    }

    public function testHasReturnsTrueForSharedInstance()
    {
        $this->container->register('one')->asSharedInstanceOf('One');
        $this->assertTrue($this->container->has('one'));
    }

    public function testMultipleSharedInstancesAreSameInstance()
    {
        $this->container->register('one')->asSharedInstanceOf('One');
        $a = $this->container->lookup('one');
        $b = $this->container->lookup('one');
        $this->assertEquals($a, $b);
    }

    public function testRegisterAndLookupArray()
    {
        $this->container->register('One')->asArray();
        $this->assertSame([], $this->container->lookup('One'));
    }

    public function testNewInstanceWithDependencies()
    {
        $this->container->register('foo')->asValue('FOO');
        $this->container->register('one')->asNewInstanceOf('One')
            ->withDependencies(['foo']);
        $obj = $this->container->lookup('one');
        $this->assertSame('FOO', $obj->arg1);
    }

    public function testNewInstanceWithMultipleDependencies()
    {
        $this->container->register('foo')->asValue('FOO');
        $this->container->register('bar')->asValue(42);
        $this->container->register('one')->asNewInstanceOf('One')
            ->withDependencies(['foo', 'bar']);
        $obj = $this->container->lookup('one');
        $this->assertSame('FOO', $obj->arg1);
        $this->assertSame(42, $obj->arg2);
    }

    public function testNewInstanceWithInjectedObjects()
    {
        $this->container->register('foo')->asValue('FOO');
        $this->container->register('one')->asNewInstanceOf('One');
        $this->container->register('two')->asNewInstanceOf('One')
            ->withDependencies(['one', 'foo']);
        $obj = $this->container->lookup('two');
        $this->assertEquals($this->container->lookup('one'), $obj->arg1);
        $this->assertSame('FOO', $obj->arg2);
    }

    public function testNewInstanceWithAddConstructorValue()
    {
        $this->container->register('one')->asNewInstanceOf('One')
            ->addConstructorValue('x')
            ->addConstructorValue(99);
        $obj = $this->container->lookup('one');
        $this->assertSame('x', $obj->arg1);
        $this->assertSame(99, $obj->arg2);
    }

    public function testNewInstanceWithAddConstructorLookup()
    {
        $this->container->register('foo')->asValue('FOO');
        $this->container->register('bar')->asValue(42);
        $this->container->register('one')->asNewInstanceOf('One')
            ->addConstructorLookup('foo')
            ->addConstructorLookup('bar');

        $obj = $this->container->lookup('one');
        $this->assertSame('FOO', $obj->arg1);
        $this->assertSame(42, $obj->arg2);
    }

    public function testResolvedDependenciesCanBeLookedUp()
    {
        $this->container->register('foo')->asValue('FOO');
        $this->container->register('one')->asNewInstanceOf('One');
        $this->container->register('two')->asNewInstanceOf('One')
            ->withDependencies(['one', 'foo']);
        $deps = $this->container->createDependenciesFor('two');
        $this->assertEquals(
            [$this->container->lookup('one'), 'FOO'], $deps
            );
    }

    public function testArrayOfDependenciesCanBeSpecified()
    {
        $this->container->register('foo')->asValue('FOO');
        $this->container->register('one')->asNewInstanceOf('One');
        $this->container->register('two')->asNewInstanceOf('One')
            ->withDependencies([['one', 'foo'], 'foo']);

        $obj = $this->container->lookup('two');
        $this->assertEquals([$this->container->lookup('one'), 'FOO'], $obj->arg1);
        $this->assertSame('FOO', $obj->arg2);
    }

    public function testArrayWithDependencies()
    {
        $this->container->register('foo')->asValue('FOO');
        $this->container->register('bar')->asValue(42);
        $this->container->register('one')->asArray('One')
            ->withDependencies(['foo', 'bar']);
        $this->assertSame(['FOO', 42], $this->container->lookup('one'));
    }

    public function testAliasCanBeSet()
    {
        $this->container->register('foo')->asValue('FOO');
        $this->container->register('bar')->asAliasOf('foo');

        $this->assertSame('FOO', $this->container->lookup('bar'));
    }

    public function testAliasOfAliasCanBeSet()
    {
        $this->container->register('foo')->asValue('FOO');
        $this->container->register('bar')->asAliasOf('foo');
        $this->container->register('zip')->asAliasOf('bar');
        $this->container->register('button')->asAliasOf('zip');

        $this->assertSame('FOO', $this->container->lookup('button'));
    }
}
                                                                                                                                                                                           <?php

class Swift_MailerTest extends \SwiftMailerTestCase
{
    public function testTransportIsStartedWhenSending()
    {
        $transport = $this->createTransport();
        $message = $this->createMessage();

        $started = false;
        $transport->shouldReceive('isStarted')
                  ->zeroOrMoreTimes()
                  ->andReturnUsing(function () use (&$started) {
                      return $started;
                  });
        $transport->shouldReceive('start')
                  ->once()
                  ->andReturnUsing(function () use (&$started) {
                      $started = true;

                      return;
                  });

        $mailer = $this->createMailer($transport);
        $mailer->send($message);
    }

    public function testTransportIsOnlyStartedOnce()
    {
        $transport = $this->createTransport();
        $message = $this->createMessage();

        $started = false;
        $transport->shouldReceive('isStarted')
                  ->zeroOrMoreTimes()
                  ->andReturnUsing(function () use (&$started) {
                      return $started;
                  });
        $transport->shouldReceive('start')
                  ->once()
                  ->andReturnUsing(function () use (&$started) {
                      $started = true;

                      return;
                  });

        $mailer = $this->createMailer($transport);
        for ($i = 0; $i < 10; ++$i) {
            $mailer->send($message);
        }
    }

    public function testMessageIsPassedToTransport()
    {
        $transport = $this->createTransport();
        $message = $this->createMessage();
        $transport->shouldReceive('send')
                  ->once()
                  ->with($message, \Mockery::any());

        $mailer = $this->createMailer($transport);
        $mailer->send($message);
    }

    public function testSendReturnsCountFromTransport()
    {
        $transport = $this->createTransport();
        $message = $this->createMessage();
        $transport->shouldReceive('send')
                  ->once()
                  ->with($message, \Mockery::any())
                  ->andReturn(57);

        $mailer = $this->createMailer($transport);
        $this->assertEquals(57, $mailer->send($message));
    }

    public function testFailedRecipientReferenceIsPassedToTransport()
    {
        $failures = [];

        $transport = $this->createTransport();
        $message = $this->createMessage();
        $transport->shouldReceive('send')
                  ->once()
                  ->with($message, $failures)
                  ->andReturn(57);

        $mailer = $this->createMailer($transport);
        $mailer->send($message, $failures);
    }

    public function testSendRecordsRfcComplianceExceptionAsEntireSendFailure()
    {
        $failures = [];

        $rfcException = new Swift_RfcComplianceException('test');
        $transport = $this->createTransport();
        $message = $this->createMessage();
        $message->shouldReceive('getTo')
                  ->once()
                  ->andReturn(['foo&invalid' => 'Foo', 'bar@valid.tld' => 'Bar']);
        $transport->shouldReceive('send')
                  ->once()
                  ->with($message, $failures)
                  ->andThrow($rfcException);

        $mailer = $this->createMailer($transport);
        $this->assertEquals(0, $mailer->send($message, $failures), '%s: Should return 0');
        $this->assertEquals(['foo&invalid', 'bar@valid.tld'], $failures, '%s: Failures should contain all addresses since the entire message failed to compile');
    }

    public function testRegisterPluginDelegatesToTransport()
    {
        $plugin = $this->createPlugin();
        $transport = $this->createTransport();
        $mailer = $this->createMailer($transport);

        $transport->shouldReceive('registerPlugin')
                  ->once()
                  ->with($plugin);

        $mailer->registerPlugin($plugin);
    }

    private function createPlugin()
    {
        return $this->getMockery('Swift_Events_EventListener')->shouldIgnoreMissing();
    }

    private function createTransport()
    {
        return $this->getMockery('Swift_Transport')->shouldIgnoreMissing();
    }

    private function createMessage()
    {
        return $this->getMockery('Swift_Mime_SimpleMessage')->shouldIgnoreMissing();
    }

    private function createMailer(Swift_Transport $transport)
    {
        return new Swift_Mailer($transport);
    }
}
