<?php

/*
 * This file is part of Psy Shell.
 *
 * (c) 2012-2018 Justin Hileman
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Psy\Test\Reflection;

use Psy\Reflection\ReflectionConstant_;

\define('Psy\\Test\\Reflection\\SOME_CONSTANT', 'yep');

class ReflectionConstantTest extends \PHPUnit\Framework\TestCase
{
    public function testConstruction()
    {
        $refl = new ReflectionConstant_('Psy\\Test\\Reflection\\SOME_CONSTANT');

        $this->assertFalse($refl->getDocComment());
        $this->assertEquals('Psy\\Test\\Reflection\\SOME_CONSTANT', $refl->getName());
        $this->assertEquals('Psy\\Test\\Reflection', $refl->getNamespaceName());
        $this->assertEquals('yep', $refl->getValue());
        $this->assertTrue($refl->inNamespace());
        $this->assertEquals('Psy\\Test\\Reflection\\SOME_CONSTANT', (string) $refl);
        $this->assertNull($refl->getFileName());
    }

    public function testBuiltInConstant()
    {
        $refl = new ReflectionConstant_('PHP_VERSION');

        $this->assertEquals('PHP_VERSION', $refl->getName());
        $this->assertEquals('PHP_VERSION', (string) $refl);
        $this->assertEquals(PHP_VERSION, $refl->getValue());
        $this->assertFalse($refl->inNamespace());
        $this->assertSame('', $refl->getNamespaceName());
    }

    /**
     * @dataProvider magicConstants
     */
    public function testIsMagicConstant($name, $is)
    {
        $this->assertEquals($is, ReflectionConstant_::isMagicConstant($name));
    }

    public function magicConstants()
    {
        return [
            ['__LINE__', true],
            ['__FILE__', true],
            ['__DIR__', true],
            ['__FUNCTION__', true],
            ['__CLASS__', true],
            ['__TRAIT__', true],
            ['__METHOD__', true],
            ['__NAMESPACE__', true],
            ['__COMPILER_HALT_OFFSET__', true],
            ['PHP_VERSION', false],
            ['PHP_EOL', false],
            ['Psy\\Test\\Reflection\\SOME_CONSTANT', false],
            ['What if it isn\'t even a valid constant name?', false],
        ];
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testUnknownConstantThrowsException()
    {
        new ReflectionConstant_('UNKNOWN_CONSTANT');
    }

    public function testExport()
    {
        $ret = ReflectionConstant_::export('Psy\\Test\\Reflection\\SOME_CONSTANT', true);
        $this->assertEquals($ret, 'Constant [ string Psy\\Test\\Reflection\\SOME_CONSTANT ] { yep }');
    }

    public function testExportOutput()
    {
        $this->expectOutputString("Constant [ string Psy\\Test\\Reflection\\SOME_CONSTANT ] { yep }\n");
        ReflectionConstant_::export('Psy\\Test\\Reflection\\SOME_CONSTANT', false);
    }

    public function testGetFileName()
    {
        $refl = new ReflectionConstant_('Psy\\Test\\Reflection\\SOME_CONSTANT');
        $this->assertNull($refl->getFileName());
    }

    /**
     * @expectedException \RuntimeException
     * @dataProvider notYetImplemented
     */
    public function testNotYetImplemented($method)
    {
        $refl = new ReflectionConstant_('Psy\\Test\\Reflection\\SOME_CONSTANT');
        $refl->$method();
    }

    public function notYetImplemented()
    {
        return [
            ['getStartLine'],
            ['getEndLine'],
        ];
    }
}
                                                                                                                              <?php

/*
 * This file is part of Psy Shell.
 *
 * (c) 2012-2018 Justin Hileman
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Psy\Test\Reflection;

use Psy\Reflection\ReflectionClassConstant;

class ReflectionClassConstantTest extends \PHPUnit\Framework\TestCase
{
    const CONSTANT_ONE = 'one';

    public function testConstruction()
    {
        $refl  = new ReflectionClassConstant($this, 'CONSTANT_ONE');
        $class = $refl->getDeclaringClass();

        $this->assertInstanceOf('ReflectionClass', $class);
        $this->assertSame('Psy\Test\Reflection\ReflectionClassConstantTest', $class->getName());
        $this->assertSame('CONSTANT_ONE', $refl->getName());
        $this->assertSame('CONSTANT_ONE', (string) $refl);
        $this->assertSame('one', $refl->getValue());
        $this->assertNull($refl->getFileName());
        $this->assertFalse($refl->getDocComment());
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testUnknownConstantThrowsException()
    {
        new ReflectionClassConstant($this, 'UNKNOWN_CONSTANT');
    }

    public function testExport()
    {
        $ret = ReflectionClassConstant::export($this, 'CONSTANT_ONE', true);
        $this->assertEquals($ret, 'Constant [ public string CONSTANT_ONE ] { one }');
    }

    public function testExportOutput()
    {
        $this->expectOutputString("Constant [ public string CONSTANT_ONE ] { one }\n");
        ReflectionClassConstant::export($this, 'CONSTANT_ONE', false);
    }

    public function testModifiers()
    {
        $refl = new ReflectionClassConstant($this, 'CONSTANT_ONE');

        $this->assertEquals(\ReflectionMethod::IS_PUBLIC, $refl->getModifiers());
        $this->assertFalse($refl->isPrivate());
        $this->assertFalse($refl->isProtected());
        $this->assertTrue($refl->isPublic());
    }

    /**
     * @expectedException \RuntimeException
     * @dataProvider notYetImplemented
     */
    public function testNotYetImplemented($method)
    {
        $refl = new ReflectionClassConstant($this, 'CONSTANT_ONE');
        $refl->$method();
    }

    public function notYetImplemented()
    {
        return [
            ['getStartLine'],
            ['getEndLine'],
        ];
    }
}
