<?php
/*
 * This file is part of PHPUnit.
 *
 * (c) Sebastian Bergmann <sebastian@phpunit.de>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace PHPUnit\Util\TestDox;

final class TestResult
{
    /**
     * @var callable
     */
    private $colorize;

    /**
     * @var string
     */
    private $testClass;

    /**
     * @var string
     */
    private $testMethod;

    /**
     * @var bool
     */
    private $testSuccesful;

    /**
     * @var string
     */
    private $symbol;

    /**
     * @var string
     */
    private $additionalInformation;

    /**
     * @var bool
     */
    private $additionalInformationVerbose;

    /**
     * @var float
     */
    private $runtime;

    public function __construct(callable $colorize, string $testClass, string $testMethod)
    {
        $this->colorize              = $colorize;
        $this->testClass             = $testClass;
        $this->testMethod            = $testMethod;
        $this->testSuccesful         = true;
        $this->symbol                = ($this->colorize)('fg-green', '✔');
        $this->additionalInformation = '';
    }

    public function isTestSuccessful(): bool
    {
        return $this->testSuccesful;
    }

    public function fail(string $symbol, string $additionalInformation, bool $additionalInformationVerbose = false): void
    {
        $this->testSuccesful                = false;
        $this->symbol                       = $symbol;
        $this->additionalInformation        = $additionalInformation;
        $this->additionalInformationVerbose = $additionalInformationVerbose;
    }

    public function setRuntime(float $runtime): void
    {
        $this->runtime = $runtime;
    }

    public function toString(?self $previousTestResult, $verbose = false): string
    {
        return \sprintf(
            "%s%s %s %s%s\n%s",
            $previousTestResult && $previousTestResult->additionalInformationPrintable($verbose) ? "\n" : '',
            $this->getClassNameHeader($previousTestResult ? $previousTestResult->testClass : null),
            $this->symbol,
            $this->testMethod,
            $verbose ? ' ' . $this->getFormattedRuntime() : '',
            $this->getFormattedAdditionalInformation($verbose)
        );
    }

    private function getClassNameHeader(?string $previousTestClass): string
    {
        $className = '';

        if ($this->testClass !== $previousTestClass) {
            if (null !== $previousTestClass) {
                $className = "\n";
            }

            $className .= \sprintf("%s\n", $this->testClass);
        }

        return $className;
    }

    private function getFormattedRuntime(): string
    {
        if ($this->runtime > 5) {
            return ($this->colorize)('fg-red', \sprintf('[%.2f ms]', $this->runtime * 1000));
        }

        if ($this->runtime > 1) {
            return ($this->colorize)('fg-yellow', \sprintf('[%.2f ms]', $this->runtime * 1000));
        }

        return \sprintf('[%.2f ms]', $this->runtime * 1000);
    }

    private function getFormattedAdditionalInformation($verbose): string
    {
        if (!$this->additionalInformationPrintable($verbose)) {
            return '';
        }

        return \sprintf(
            "   │\n%s\n",
            \implode(
                "\n",
                \array_map(
                    function (string $text) {
                        return \sprintf('   │ %s', $text);
                    },
                    \explode("\n", $this->additionalInformation)
                )
            )
        );
    }

    private function additionalInformationPrintable(bool $verbose): bool
    {
        if ($this->additionalInformation === '') {
            return false;
        }

        if ($this->additionalInformationVerbose && !$verbose) {
            return false;
        }

        return true;
    }
}
                                                                                                                        <?php
/*
 * This file is part of PHPUnit.
 *
 * (c) Sebastian Bergmann <sebastian@phpunit.de>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace PHPUnit\Util\TestDox;

/**
 * Prints TestDox documentation in HTML format.
 */
final class HtmlResultPrinter extends ResultPrinter
{
    /**
     * @var string
     */
    private const PAGE_HEADER = <<<EOT
<!doctype html>
<html lang="en">
    <head>
        <meta charset="utf-8"/>
        <title>Test Documentation</title>
        <style>
            body {
                text-rendering: optimizeLegibility;
                font-variant-ligatures: common-ligatures;
                font-kerning: normal;
                margin-left: 2em;
            }

            body > ul > li {
                font-family: Source Serif Pro, PT Sans, Trebuchet MS, Helvetica, Arial;
                font-size: 2em;
            }

            h2 {
                font-family: Tahoma, Helvetica, Arial;
                font-size: 3em;
            }

            ul {
                list-style: none;
                margin-bottom: 1em;
            }
        </style>
    </head>
    <body>
EOT;

    /**
     * @var string
     */
    private const CLASS_HEADER = <<<EOT

        <h2 id="%s">%s</h2>
        <ul>

EOT;

    /**
     * @var string
     */
    private const CLASS_FOOTER = <<<EOT
        </ul>
EOT;

    /**
     * @var string
     */
    private const PAGE_FOOTER = <<<EOT

    </body>
</html>
EOT;

    /**
     * Handler for 'start run' event.
     */
    protected function startRun(): void
    {
        $this->write(self::PAGE_HEADER);
    }

    /**
     * Handler for 'start class' event.
     */
    protected function startClass(string $name): void
    {
        $this->write(
            \sprintf(
                self::CLASS_HEADER,
                $name,
                $this->currentTestClassPrettified
            )
        );
    }

    /**
     * Handler for 'on test' event.
     */
    protected function onTest($name, bool $success = true): void
    {
        $this->write(
            \sprintf(
                "            <li style=\"color: %s;\">%s %s</li>\n",
                $success ? '#555753' : '#ef2929',
                $success ? '✓' : '❌',
                $name
            )
        );
    }

    /**
     * Handler for 'end class' event.
     */
    protected function endClass(string $name): void
    {
        $this->write(self::CLASS_FOOTER);
    }

    /**
     * Handler for 'end run' event.
     */
    protected function endRun(): void
    {
        $this->write(self::PAGE_FOOTER);
    }
}
